/**
 * \file: ald_types.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#ifndef ALD_TYPES_H_
#define ALD_TYPES_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdbool.h>
#include <stdint.h>

#include <sys/time.h>

//----------------------------------------------- global macros -------------------------------------------------------
#define ARRAY_SIZE(x)	(sizeof(x) / sizeof(x[0]))

//----------------------------------------------- global constants ----------------------------------------------------

#define LEVEL_SUBDIR						"/security_levels"
#define LEVEL_CONF_FN						"level.conf"

#define PUBLIC_KEY_EXT						".pub"
#define PUBLIC_KEY_WRP_EXT					".pub.wrap"
#define PRIVATE_KEY_EXT						".priv"
#define SIGNATURE_EXT						".sign"

#define PRIV_KEY_LEN_BITS					2048
#define SIG_ALGORITHM 						EVP_sha256()


// public keys for levels are named <LEVEL_KEY_NAME><PUBLIC_KEY_EXT> and are placed in the respective folder of the level
// private keys for levels are name <LEVEL_KEY_NAME>_<level_number><PRIVATE_KEY_EXT> and are placed in
// location passed to ald_init
#define LEVEL_KEY_NAME						"level"

//public key for signature data base and script signature generation is named <db_file_name><PUBLIC_KEY_EXT> and placed at
//the same location where the data base is to be stored
//private key for signature data base and script signature generation is named <SIGDB_PRIV_KEY_NAME><PRIVATE_KEY_EXT> and placed
//at the location passed to ald_init
#define SIGDB_PRIV_KEY_NAME					"signature_db" PRIVATE_KEY_EXT

#define PREPARE_LEVEL_CHANGE_FN		"prepare_change.sh"
#define FINALIZE_LEVEL_CHANGE_FN		"finalize_change.sh"
#define PREPARE_LEVEL_CHANGE_RECO_FN		"prepare_recovery.sh"
#define FINALIZE_LEVEL_CHANGE_RECO_FN		"finalize_recovery.sh"

#define TRANSIENT_LEVEL_TAG					"[transient level]"
#define PERMANENT_LEVEL_TAG					"[permanent level]"

#define FEATURE_SUBDIR						"/security_features"
#define SCRIPT_EXTN							".sh"
#define FEATURE_ENABLE_PERM_FN				"/enable_permanent"SCRIPT_EXTN
#define FEATURE_DISABLE_PERM_FN				"/disable_permanent"SCRIPT_EXTN
#define FEATURE_ENABLE_VOLA_FN				"/enable_volatile"SCRIPT_EXTN
#define FEATURE_DISABLE_VOLA_FN				"/disable_volatile"SCRIPT_EXTN
#define FEATURE_ENABLE_PERM_DIR				"enable_permanent"
#define FEATURE_DISABLE_PERM_DIR			"disable_permanent"
#define FEATURE_ENABLE_VOLA_DIR				"enable_volatile"
#define FEATURE_DISABLE_VOLA_DIR			"disable_volatile"


//----------------------------------------------- global data types ---------------------------------------------------

//items of the challenge data structure
#define CHALLENGE_RANDOM_DATA_SIZE					16
#define CHALLENGE_CHIP_ID_SIZE						8
#define CHALLENGE_KEY_SET_ID_SIZE					4
#define CHALLENGE_ECU_ID_SIZE						16
#define CHALLENGE_RESERVED_DATA_SIZE				40

//items of the response data structure
#define RESPONSE_SERIAL_NUMBER_SIZE					25
#define RESPONSE_USER_PUBKEY_SIZE					512
#define RESPONSE_RESERVED_DATA_SIZE					247
#define RESPONSE_SIGNATURE_SIZE_MAX					512
#define RESPONSE_SIGNATURE_SIZE_USED				256

#define PROC_CHIP_ID_KEY							"Serial"

#define ERRMEM_DEVICE								"/dev/errmem"

#define CLOSED_LEVEL                                0
#define FAIL_SAFE_LEVEL                             CLOSED_LEVEL
#define MAX_LEVEL                                   10000 /* Max valid ALD level - min level is 0 */

#define ALD_PROTOCOL_MAJOR_VERSION					1
#define ALD_PROTOCOL_MINOR_VERSION					0

/**
 * \brief the data type that defines the security level
 */
typedef uint32_t security_level_t;

/**
 * \brief Defines types of a level change.
 */
typedef enum change_type_t
{
	/**
	 * \brief a level change has been requested
	 */
	ALD_NORMAL_CHANGE=0,

	/**
	 * \brief a level change is performed as recovery action
	 */
	ALD_RECOVERY_CHANGE=1,

	/**
	 * \brief a recovery action is performed as the script execution timed out
	 *  Going back to previous Level
	 */
	ALD_RECOVERY_ON_TIMEOUT

} __attribute__((aligned(4),packed)) change_type_t;

/**
 * defines states of the daemon state machine with respect to level change handling
 */
typedef enum daemon_level_state_t
{
	/**
	 * \brief a transient state entered during startup
	 */
	DAEMON_LEVEL_INITIALIZATION=0,

	/**
	 * \brief Nothing ongoing
	 */
	DAEMON_LEVEL_STABLE,

	/**
	 * \brief A recovery sequence to complete a permanent level change has been initiated
	 */
	DAEMON_LEVEL_RECOVERY_ONGOING,

	/**
	 * \brief A level change sequence has been started
	 */
	DAEMON_LEVEL_CHANGE_ONGOING,

	/**
	 * \brief Can't process a level change as RECOVERY failed and waiting for RECOVERY on startup to heal
	 */
	DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED

} __attribute__((aligned(4))) daemon_level_state_t;

/**
 * defines states of the daemon state machine with respect to generic state of the service
 */
typedef enum daemon_service_state_t
{
	/**
	 * \brief a transient state entered during startup
	 */
	DAEMON_SERVICE_INITIALIZATION=0,

	/**
	 * \brief Recovery or level change during startup completed or not needed - but still waiting for  APP IFACE initialization
	 */
	DAEMON_SERVICE_WAITING_FOR_APP_IFACE_READY,

	/**
	 * \brief APP IFACE initialization complete - but still waiting for recovery, level change during startup or check that neither one is required
	 */
	DAEMON_SERVICE_WAITING_FOR_LEVEL_CHANGE_RECOVERY_ON_STARTUP,

	/**
	 * \brief Setup of daemon is completed
	 */
	DAEMON_SERVICE_READY,

	/**
	 * \brief Entered on a shutdown request received while processing a level change to wait for the change to finish
	 */
	DAEMON_SERVICE_SHUTDOWN_PENDING,

} __attribute__((aligned(4))) daemon_service_state_t;


/**
 * \brief defines log levels the daemon is running in
 */
typedef enum logger_loglevel_t
{
	/**
	 * \brief Nothing is logged at all
	 */
	LOGGER_LEVEL_NONE=0,

	/**
	 * \brief Only errors are logged
	 */
	LOGGER_LEVEL_ERROR,

	/**
	 * \brief Error and info message are logged.
	 */
	LOGGER_LEVEL_INFO,

	/**
	 * \brief Error, info, and debug message are logged.
	 */
	LOGGER_LEVEL_DEBUG
} __attribute__((aligned(4))) logger_loglevel_t;

/**
 * \brief defines key verification behaviour while solving challenge in level change request
 */
typedef enum key_check_behavior_t
{
	/*
	 * Unlock all the levels below the given level using the given key
	 */
	UNLOCK_THAT_AND_BELOW_LEVELS = 0,

	/*
	 * Unlock only the level corresponding to the given key
	 */
	UNLOCK_ONLY_THAT_LEVEL = 1

} __attribute__((aligned(4), packed)) key_check_behavior_t;

/**
 * \brief defines the different kind of configuration file items
 */
typedef enum cgf_item_kind_t
{
	/**
	 * The variable is of string kind
	 */
	CFG_STRING,

	/**
	 * The variable is of integer kind
	 */
	CFG_UINT32,

	/**
	 * The variable states different log options
	 */
	CFG_LOGLEVEL,

	/**
	 * The variable kind used for display items like help
	 */
	CFG_FLAG,

	/**
	 * The variable states different challenge verify key behaviour option
	 */
	CFG_KEY_BEHAVIOR

} __attribute__((aligned(4), packed)) cgf_item_kind_t;

/**
 * \brief defines the data type of value of the configuration file item
 */
typedef union cfg_val_t
{
	/**
	 * The variable value is of string type
	 */
	char *str_val;

	/**
	 * The variable value is of integer type
	 */
	uint32_t u32_val;

	/**
	 * The variable value is of logger_loglevel_t type
	 */
	logger_loglevel_t log_level;

} __attribute__((aligned(4), packed)) cfg_val_t;

/**
 * \brief defines the specification of configuration file items
 */
typedef struct cfg_item_spec_t
{
	/**
	 * config file item's kind
	 */
	cgf_item_kind_t kind;

	/**
	 * config file item's key detail
	 * key in configuration file - NULL if no config entry
	 */
	char *conf_key;

	/**
	 * config file item's default value
	 */
	const cfg_val_t default_val;

} __attribute__((aligned(4), packed)) cfg_item_spec_t;

/**
 * \brief defines the cmdline argument options of configuration file items
 */
typedef struct cfg_cmd_line_spec_t
{
	/**
	 * long form of the cmdline argument to a config file item
	 */
	char *cmd_arg_long;

	/**
	 * long form of the cmdline argument to a config file item
	 */
	char *cmd_arg_short;

} __attribute__((aligned(4), packed)) cfg_cmd_line_spec_t;

/**
 * \brief defines attributes of configuration file items
 */
typedef struct cfg_item_t
{
	/**
	 * the item specification
	 */
	const cfg_item_spec_t *spec;

	/**
	 * the item cmdline arg options
	 */
	const cfg_cmd_line_spec_t *cmdline;

	/**
	 * set is true when the item is parsed from cmdline or config file
	 * cfg_val_t val != NULL
	 */
	bool set;

	/**
	 * the value of item parsed from either cmdline or config file
	 */
	cfg_val_t val;

} __attribute__((aligned(4), packed)) cfg_item_t;

/**
 * \brief Defines a data type that represents a challenge
 */
typedef struct challenge_t
{
	/**
	 * A random number generated by the daemon
	 */
	char random_data[CHALLENGE_RANDOM_DATA_SIZE];

	/**
	 * The seconds of the time stamp at which the challenge was created
	 */
	uint32_t tv_sec;

	/**
	 * The useconds of the time stamp at which the challenge was created
	 */
	uint32_t tv_usec;

	/**
	 * the chip id of the head unit on which the daemon is running
	 */
	char chip_id[CHALLENGE_CHIP_ID_SIZE];

	/**
	 * a key set id
	 */
	char key_set_id[CHALLENGE_KEY_SET_ID_SIZE];

	/**
	 * The ECU ID passed when requesting the challenge
	 */
	char ecu_id[CHALLENGE_ECU_ID_SIZE];

	/**
	 * The minor version number of ALD challenge/response protocol
	 */
	uint16_t ALD_protocol_major;

	/**
	 * The minor version number of ALD challenge/response protocol
	 */
	uint16_t ALD_protocol_minor;

	/**
	 * reserved data
	 */
	char reserved[CHALLENGE_RESERVED_DATA_SIZE];

} __attribute__((aligned(4), packed)) challenge_t;

/**
 * \brief Defines a data type that represents a challenge response
 * The challenge response is sent to the daemon to request a level change.
 */
typedef struct challenge_response_t
{
	/**
	 * \brief the challenge requested from the daemon before
	 */
	challenge_t challenge;

	/**
	 * a serial number
	 */
	char serial_number[RESPONSE_SERIAL_NUMBER_SIZE];

	/**
	 * \brief the targeted security level
	 */
	security_level_t targeted_level;

	/**
	 * \brief I've no bloody idea what the hack this is for ...
	 */
	char user_pubkey[RESPONSE_USER_PUBKEY_SIZE];

	/**
	 * ALD challenge/response protocol major version number of ALD client
	 */
	uint16_t client_ALD_protocol_major;

	/**
	 * ALD challenge/response protocol minor version number of ALD client
	 */
	uint16_t client_ALD_protocol_minor;

	/**
	 * \brief I've no bloody idea what the hack this is for ...
	 */
	char reserved[RESPONSE_RESERVED_DATA_SIZE];

	/**
	 * the signature
	 */
	char signature[RESPONSE_SIGNATURE_SIZE_MAX];
} __attribute__((aligned(4), packed)) challenge_response_t;

/**
 * \brief defines error codes used within the daemon.
 */
typedef enum error_code_t
{
	/**
	 * \brief no failures
	 */
	RESULT_OK									=	0,

	/**
	 * In general: A function has been used in an invalid state or invalid manner
	 */
	RESULT_INVALID								=	1,

	/**
	 * \brief Return in case of any kind of resource issues (such as memory problems).
	 */
	RESULT_NORESOURCE							=	2,

	/**
	 * The file in which the persistent state is stored or should be stored can not be opened from the configured location.
	 */
	RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES	=	3,

	/**
	 * A level change is requested to an invalid level
	 */
	RESULT_INVALID_LEVEL						=	4,

	/**
	 * feature scripts or level configuration files do not fit to the signatures stored in the signature db
	 */
	RESULT_SCRIPTS_MODIFIED						=	5,

	/**
	 * one or more scripts executed during a level change returned a non zero exit code
	 */
	RESULT_SCRIPTS_FAILED						=	6,

	/**
	 * one or more scripts executed during a level change returned a non zero exit code
	 */
	RESULT_SCRIPTS_TIMEOUT						=	7,

	/**
	 * The configuration of the requested level was modified
	 */
	RESULT_LEVEL_COMFIGURATION_MODIFIED				=	8,

	/**
	 * The key to check the signature was modified
	 */
	RESULT_VERIFICATION_KEY_MODIFIED			=	9,

	/**
	 * A challenge for a request has not been solved correctly
	 */
	RESULT_CHALLENGE_INVALID_SIGNATURE			=	10,

	/**
	 * A challenge for a request has not been solved correctly
	 */
	RESULT_CHALLENGE_EXPIRED					=	11,

	/**
	 * A challenge for a request has not been solved correctly
	 */
	RESULT_CHALLENGE_INVALID_RESPONSE			=	12,

	/**
	 * a level change request is rejected because another change request is currently processed
	 */
	RESULT_DAEMON_BUSY_LEVEL_CHANGE				=	13,

	/**
	 * a level change request is rejected because a recovery is currently processed
	 */
	RESULT_DAEMON_BUSY_LEVEL_RECOVERY			=	14,

	/**
	 * the script exceution timed out
	 */
	RESULT_SCRIPT_EXEC_TIMEOUT					=	15,

	/**
	 * a level change request is rejected, as ALD is waiting for a recovery on startup to heal.
	 */
	RESULT_DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED			=	16,

	/**
	 * verification of a file our data block against a signature failed
	 */
	RESULT_VERIFICATION_FAILED,

	/**
	 * \brief the daemon was started with invalid command line options
	 */
	RESULT_INVALID_ARGS,

	/**
	 * \brief the daemon just printed out command line help and exits
	 */
	RESULT_HELP_PRINTED,

	/**
	 * \brief the application interface had some problems connecting to the dbus system bus
	 */
	RESULT_DBUS_SYSTEM_BUS_ERROR,

	/**
	 * The Wrapped file is corrupted.
	 */
	RESULT_WRAPPED_FILE_CORRUPT,

	/**
	 * The configuration of the requested level is invalid
	 */
	RESULT_INVALID_LEVEL_CONFIGURATION,

	/**
	 * an invalid signature db has been detected
	 */
	RESULT_SIGNATURE_DB_INVALID,

	/**
	 * the signature db file could not be created
	 */
	RESULT_SIGNATURE_DB_CREATION_FAILED,

	/**
	 * Mmm, somehow self explaining ...
	 */
	RESULT_FILE_NOT_FOUND,

	/**
	 * an error occured during private/public key generation
	 */
	RESULT_OPENSSL_KEYGEN_FAILED,

	/**
	 * unable to call openssl to generate new private/public key pais
	 */
	RESULT_OPENSSL_MISSING,

	/**
	 * a private or public key read in from file is invalid
	 */
	RESULT_INVALID_KEY,

	/**
	 * the signing of a file failed
	 */
	RESULT_SIGNING_FAILED,

	/**
	 * the signature that is tried to be read out from file is invalid
	 */
	RESULT_SIGNATURE_FILE_INVALID,

	/**
	 * the en/decryption failed
	 */
	RESULT_CRYPTOGRAPHIC_OPERATION_FAILED,

	/**
	 * Invalid file type
	 */
	RESULT_INVALID_FILE_TYPE,

	/**
	 * Mandatory file not found,this will result is failure of function
	 */
	RESULT_MUST_FILE_NOT_FOUND,

	/*
	 * Stat error
	 */
	RESULT_STAT_ERROR,

    /*
     * Command is invalid error
     */
	RESULT_INVALID_CMD

} __attribute__((aligned(4),packed)) error_code_t;

/**
 * \brief Defines a data type for various vectors of a command in general
 */
typedef struct command_vtable_t
{
	/**
	 * Command name
	 */
	const char *command;

	/**
	 * A short description about the command
	 */
	const char *command_description;

	/**
	 * Function pointer pointing to initialization call of the command
	 */
	error_code_t (*init)(void);

	/**
	 * Function pointer pointing to parsing call of the command
	 */
	error_code_t (*parse_args)(const char *binary_name, int argc, char* argv[]);

	/**
	 * Function pointer pointing to main call of the command
	 */
	error_code_t (*start)(void);

	/**
	 * Function pointer pointing to de-initialization call of the command
	 */
	void (*deinit)(void);
} __attribute__((aligned(4), packed)) command_vtable_t;

#endif
